/*
|============================================================================
|
|       Copyright (C) 2011 ProSoft Technology. All rights reserved.
|
|  File:             upgrademanager.c
|
|  Class(es):        
|
|  Inherits From:    
|
|  Summary:          
|
|  Project(s):       PLX Utility
|
|  Subsystem:        Common
|
|  Contributors:     Henry Yu(HYU)
|
|  Description:      
|
|  Notes:            In order for this program to work, it MUST be set with 
|                    the SETUID permission bit enabled and the owner of the 
|                    file must be root. This is due to the need of directly 
|                    using /dev/mem.
|
|============================================================================
|  Version     Date     Author  Change    Description
|----------------------------------------------------------------------------
|  Build    01/26/2011  Chientan add set time support, read time after write time to avoid lock up.
|           09/02/2011  HYU     Migrated to ProLinx Linux platform.
|           11/30/2011  HYU     Modified cgiMain to remove the call of exit so that cgic can free the resources.
|           03/30/2012  HYU     Modified cgiMain to replace "exit" with "return".
|                               Fixed the issue "Duplicate headers received from server" for Chrome browser.
|                               Fixed the issue "Corrupted Content Error" for Firefox browser.
|			11/30/2012  PJB		Modified for LDM support, reading jumpers, using less stack.
|			07/31/2013	PJB		Removed security decoding.
|			12-30-2014  JOA 	   Modified to check setup jumper when module rescue function is used.
============================================================================
*/
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <signal.h>
#include <time.h>
#include <sys/time.h>
#include <errno.h>
#include <termios.h>
#include <sys/ioctl.h>
#include <sys/time.h>
#include <linux/rtc.h>
#include <sys/mman.h>
#include "cgic.h"
#include "firmwareheader.h"

#include "mvi69api.h"




// This define defines the maximum posible upload size
// IZI 05/11/2009 Increased size of firmware file to 15M from 5M
#define MAX_UPLOAD_SIZE 15000000L

// This defines where the compresed file will be stored after download
#define UPLOAD_TEMPORAL_FILENAME     "/psfttmp/update.tar.gz"

#define PROSOFT_STARTUP_SCRIPT       "/psft/bin/startup"

#define UPLOAD_STATUS_FILE           "/www/html/uploadstatus"


#define UPLOAD_OK              0
#define UPLOAD_INTERNAL_ERROR -1
#define UPLOAD_INVALID_FILE    1


/*************************************************************************************************************
/ Purpose: Function that writes the time to the RTC
/
/ Params: struct tm: Time to write to the RTC
/
/ Returns: 0 if successful
/          -1 on error
/
/ Comments:
/
***********************************************************************************************************/
int setrtc(struct tm *time)
{
	int rtcfd = open("/dev/rtc0",O_RDWR);
	if (rtcfd<0)
	{
		perror("Error Opening /dev/rtc0");
		return -1;
	}

	if (ioctl(rtcfd,RTC_SET_TIME,time)<0)
	{
		perror("Error in RTC_SET_TIME");
		close(rtcfd);
		return -1;
	}

	close(rtcfd);

	return 0;
}


/*****************************************************************************
 * Purpose: Function that receives an uploaded prosoft firmware file if the
 *          file is not valid the transfer is aborted.
 *
 * Params: int debug: if 0 no debug is output.
 *
 * Returns:   0 (UPLOAD_OK) if successful
 *           -1 (UPLOAD_INTERNAL_ERROR) on internal errors
 *            1 (UPLOAD_INVALID_FILE) Invalid firmware file
 *
 * Comments:
 *
 *
******************************************************************************/
void writeStatusFile(char *statusString)
{
    // Create the upload status file
    FILE *statusFile = fopen(UPLOAD_STATUS_FILE,"w");
    fprintf(statusFile,"{STATUS:\"%s\"}",statusString);
    fclose(statusFile);
    chmod(UPLOAD_STATUS_FILE,0666);
}

#define BUFFER_SIZE 16384
char buffer[BUFFER_SIZE];

char name[1024];
unsigned char digest[20];


/*****************************************************************************
 * Purpose: Function that receives an uploaded prosoft firmware file if the
 *          file is not valid the transfer is aborted.
 *
 * Params: int debug: if 0 no debug is output.
 *
 * Returns:   0 (UPLOAD_OK) if successful
 *           -1 (UPLOAD_INTERNAL_ERROR) on internal errors
 *            1 (UPLOAD_INVALID_FILE) Invalid firmware file
 *
 * Comments:
 *
 *
******************************************************************************/
int uploadProsoftFirmware(int debug)
{
	// file received from the browser
	cgiFilePtr inFilePtr;



	int size;


	// Get the form filename. If we don't get it it means that nothing
	// was uploaded
	if (cgiFormFileName("file", name, sizeof(name)) != cgiFormSuccess) {
		if (debug!=0) fprintf(cgiOut, "<p>No file was uploaded.<p>\n");
        writeStatusFile("ERROR");
		return UPLOAD_INVALID_FILE;
	}


	// Make sure only small files are uploaded
	cgiFormFileSize("file", &size);
	if (size>MAX_UPLOAD_SIZE) {
		if (debug!=0) fprintf(cgiOut, "<p>File too Large.<p>\n");
        writeStatusFile("ERROR");
		return UPLOAD_INVALID_FILE;
	}


	// The form file is opened (so we can read the file that the browser is sending
	if (cgiFormFileOpen("file", &inFilePtr) != cgiFormSuccess) {
		if (debug!=0) fprintf(cgiOut, "Could not open the input file.<p>\n");
        writeStatusFile("ERROR");
		return UPLOAD_INVALID_FILE;
	}

	int fdOutFile;

	if ((fdOutFile = open(UPLOAD_TEMPORAL_FILENAME,O_WRONLY | O_CREAT | O_TRUNC))<0) {
		if (debug!=0) fprintf(cgiOut, "Error Creating Output File: \"%s\"\n",UPLOAD_TEMPORAL_FILENAME);
		cgiFormFileClose(inFilePtr);
        writeStatusFile("ERROR");
		return UPLOAD_INTERNAL_ERROR;
	}
	fchmod(fdOutFile,S_IRUSR | S_IWUSR | S_IRGRP | S_IWGRP);


    int readLen = 0;
	while (cgiFormFileRead(inFilePtr, buffer, BUFFER_SIZE, &readLen) == cgiFormSuccess)
	{
		write(fdOutFile,buffer,readLen);
	}

	cgiFormFileClose(inFilePtr);
	close(fdOutFile);

    writeStatusFile("OK");
	return UPLOAD_OK;

}




/*****************************************************************************
 * Purpose: Function that orders the linux system to shutdown
 *
 * Params: Nothing
 *
 * Returns: 0 if OK
 *
 * Comments:  At this moment this function is implemented calling the linux
 *            shutdown program. A more optimized way of doing it is advised
 *
 *
******************************************************************************/
int shutdownSystem( void )
{

	signal(SIGCLD, SIG_IGN);  // now I don't have to wait()
	pid_t pid;

	switch(pid=fork()) {
		case -1: // Something went wrong
			perror("fork");
			return -1;
		case 0:  // this is the child process
			//sleep(4); // Wait so at least this page can be returned
			// In order to be able to do a shutdown the process has to be root
			// because the web server doesn't run as root, the suid bit has to be added
			// to the plx80upgrade
			setuid(geteuid());
			// Kill init process
			kill(1,SIGTERM);
            exit(0);
		default: // This is the parent process
			return 0;
	}
	return(1);
}


/*****************************************************************************
 * Purpose: Function that executes the Prosoft startup script to execute an
 *          update request
 *
 * Params: Nothing
 *
 * Returns: 0 if OK
 *
 * Comments:
 *
******************************************************************************/
int shutdownCommDriver( void )
{
	signal(SIGCLD, SIG_IGN);  // now I don't have to wait()
	pid_t pid;

	switch(pid=fork()) {
		case -1: // Something went wrong
			perror("fork");
			return -1;
		case 0:  // this is the child process
			// sleep is not required because of the long time it takes to shutdown
			//sleep(4); // Wait so at least this page can be returned
			// In order to be able to do a shutdown the process has to be root
			// because the web server doesn't run as root, the suid bit has to be added
			// to the /psft/httpd/cgi-bin/upgrademanager
			setuid(geteuid());
			execlp(PROSOFT_STARTUP_SCRIPT,PROSOFT_STARTUP_SCRIPT,"update",(char *)NULL);
            exit(0);
		default: // This is the parent process
			return 0;
	}
	return(1);
}





/*****************************************************************************
 * Purpose: Function that determines if the module is running in setup mode
 *
 * Params: Nothing
 *
 * Returns: 0 if not in setup mode
 *          1 if setup jumper is installed
 *
 * Comments:
 *
******************************************************************************/
int inSetupMode( void )
{

    MVI69HANDLE handle;
    int result;

    //Open the CIP BP driver without BP access.
    if (MVI69_OpenNB(&handle) != MVI69_SUCCESS)
        return 0;

    result = 0;
    MVI69_GetSetupJumper(handle, &result);

    MVI69_Close(handle);

    return result;
}

/*****************************************************************************
 * Purpose: Restores the module to the factory default state.
 *
 * Params: Nothing
 *
 * Returns: 0 if OK
 *
 * Comments:
 *
 *
******************************************************************************/
// defines used by the rescue function
#define	BACKUP_SYSTEM_IMAGE	 		"/backup/systemrestore.tgz"
#define PSFT_UPDATE_FILE	 		"/psfttmp/update.tgz"
#define BACKUP_SYSTEM_RESTORE_CMD 	"tar -xzf /backup/systemrestore.tgz -C /"
#define REMOVE_ALL_STARTUP_SCRIPTS  "rm -rf /etc/init.d/*"

int RescueModule( void )
{
	int				result;
	struct stat		file_stat;
	char			error_string[100];


	// make sure the setup jumper is installed.  it should already have been checked.

	// make sure that the systemrestore.tgz is present and is a regular file
	if (  ( stat( BACKUP_SYSTEM_IMAGE, &file_stat ) != -1 )
			&& ( (file_stat.st_mode & S_IFMT) == S_IFREG ) )
	{
		// remove the startup scripts, since extras could be messing things up
		system( REMOVE_ALL_STARTUP_SCRIPTS );

		// removed the prosoft update file, just in case it was left around.
		result = unlink( PSFT_UPDATE_FILE );
		if ( (result != 0) && (errno != ENOENT) )
		{
			sprintf( error_string, "RESCUE_ERROR: rm update file: %d", errno );
			writeStatusFile( error_string );
		}
		else
		{
		// execute a tar extraction

		system( BACKUP_SYSTEM_RESTORE_CMD );
		}

		result = 0;

		writeStatusFile("RESCUED");
	}
	else
	{
		result = -1;
        writeStatusFile("RESCUE_ERROR");
	}

	return( result );
}
/*****************************************************************************
 * Purpose: Function that converts field value to a tv time
 *
 * Params: tv struct to place time in.
 *
 * Returns: 0 if time convert successful
 * 			-1 if there is an error
 *
 * Comments:
 *
******************************************************************************/
int getDateAndTime(	struct timeval *tv)
{
	const int MAX_FIELD_SIZE = 20;
	char fieldValue[MAX_FIELD_SIZE+1];
	struct tm newTime;
	time_t timet;


	cgiFormStringNoNewlines( "newYear", fieldValue, MAX_FIELD_SIZE);
	newTime.tm_year = atoi(fieldValue) - 1900;

	// an invalid month is setup
	newTime.tm_mon = 12;

	cgiFormStringNoNewlines( "newMonth", fieldValue, MAX_FIELD_SIZE);
	if (strcasecmp(fieldValue,"January")==0) newTime.tm_mon = 0;
	if (strcasecmp(fieldValue,"February")==0) newTime.tm_mon = 1;
	if (strcasecmp(fieldValue,"March")==0) newTime.tm_mon = 2;
	if (strcasecmp(fieldValue,"April")==0) newTime.tm_mon = 3;
	if (strcasecmp(fieldValue,"May")==0) newTime.tm_mon = 4;
	if (strcasecmp(fieldValue,"June")==0) newTime.tm_mon = 5;
	if (strcasecmp(fieldValue,"July")==0) newTime.tm_mon = 6;
	if (strcasecmp(fieldValue,"August")==0) newTime.tm_mon = 7;
	if (strcasecmp(fieldValue,"September")==0) newTime.tm_mon = 8;
	if (strcasecmp(fieldValue,"October")==0) newTime.tm_mon = 9;
	if (strcasecmp(fieldValue,"November")==0) newTime.tm_mon = 10;
	if (strcasecmp(fieldValue,"December")==0) newTime.tm_mon = 11;

	cgiFormStringNoNewlines("newDay", fieldValue, MAX_FIELD_SIZE);
	newTime.tm_mday = atoi(fieldValue);

	cgiFormStringNoNewlines( "newHours", fieldValue, MAX_FIELD_SIZE);
	newTime.tm_hour = atoi(fieldValue);

	cgiFormStringNoNewlines( "newMinutes", fieldValue, MAX_FIELD_SIZE);
	newTime.tm_min = atoi(fieldValue);

	cgiFormStringNoNewlines( "newSeconds", fieldValue, MAX_FIELD_SIZE);
	newTime.tm_sec = atoi(fieldValue);

	newTime.tm_isdst = -1;

	// Check the validity
	if 	((newTime.tm_year<0) || (newTime.tm_year>200)) return -1;
	if 	((newTime.tm_mon<0) || (newTime.tm_mon>11)) return -1;
	if 	((newTime.tm_mday<0) || (newTime.tm_mday>31)) return -1;
	if 	((newTime.tm_hour<0) || (newTime.tm_hour>23)) return -1;
	if 	((newTime.tm_min<0) || (newTime.tm_min>59)) return -1;
	if 	((newTime.tm_sec<0) || (newTime.tm_sec>59)) return -1;

	timet = mktime(&newTime);
	if (timet<0) return -1;

	tv->tv_sec = timet;
	tv->tv_usec = 0;

	return 0;
}

/*****************************************************************************
 * Purpose: Function to set date and time to the module
 *
 * Params: 	struct timeval tv : Value that has to be used the time to
 *
 * Returns: 0 if OK
 *
 * Comments: The reason that this function returns the time instead of setting
 *           the time by itself is that BOA uses current time to determine
 *           timeouts. If the time is changed the request is automatically timed
 *           out. For this reason buffers have to be flushed before changing the
 *           time.
 *
******************************************************************************/
int setDateAndTime( void )
{
    time_t newtime;
    struct tm newHwTime;
    struct timeval tv;
    
    if (getDateAndTime(&tv) != 0)
    {
        return -1;
    }

    cgiHeaderLocation("../index.html");
    fflush(NULL);
    
    setuid(geteuid());
    
    settimeofday(&tv, NULL);
    
    //After update system time, we should set the hardware RTC time.
    newtime=time(NULL);
    gmtime_r(&newtime,&newHwTime);
    setrtc(&newHwTime);

    return 0;
}


int cgiMain( void )
{


	/* Check for QUERY_STRING first. This is used in requests executed in this way:
	 * http://xxx.xxx.xxx.xxx/services/upgrademanager?function
	 * in this way we don't need to put a FORM just a link with that format
	 */
	if (strcmp(cgiQueryString,"upload")==0) {
		if (inSetupMode()) {
			cgiHeaderLocation("../startupgrade.html");
		} else {
			cgiHeaderLocation("../upgradeerror_nosetup.html");
		}
		return (0);
	}

	/* If a submit button has already been clicked, act on the
		submission of the form. */
	if (cgiFormSubmitClicked("AbortStartUpgrade") == cgiFormSuccess) {
		// Aborts a Start Upgrade. The upgrade file is erased as security but
		// the upgrade file shouldn't exist yet
		unlink(UPLOAD_TEMPORAL_FILENAME);
		cgiHeaderLocation("../index.html");
	} else if (cgiFormSubmitClicked("ContinueReflash") == cgiFormSuccess) {
		// Stops the executing driver and redirects to the software
		// upload page
		if (inSetupMode()) {
			//shutdownCommDriver();  TODO: FIX With the right shutdwon Comm Driver
			unlink(UPLOAD_TEMPORAL_FILENAME);
			unlink(UPLOAD_STATUS_FILE);
			cgiHeaderLocation("../upgrade.html");
		} else {
			cgiHeaderLocation("../upgradeerror_nosetup.html");
		}
	} else 	if (cgiFormSubmitClicked("UploadFirmware") == cgiFormSuccess) {
		if (inSetupMode()) {
			if (uploadProsoftFirmware(0)==UPLOAD_OK) {
				cgiHeaderLocation("../upgradeok.html");
			} else {
				cgiHeaderLocation("../upgradeerror.html");
			}
		} else {
			cgiHeaderLocation("../upgradeerror_nosetup.html");
		}
	} else 	if (cgiFormSubmitClicked("ResetModule") == cgiFormSuccess) {
		if (shutdownSystem()!=0) {
			cgiHeaderLocation("../upgradeerror.html");
		}
	} else 	if (cgiFormSubmitClicked("AbortUpload") == cgiFormSuccess) {
		unlink(UPLOAD_TEMPORAL_FILENAME);
		if (shutdownSystem()!=0) {
			cgiHeaderLocation("../upgradeerror.html");
		}
	} else if (cgiFormSubmitClicked("updatedatetime") == cgiFormSuccess) {
		if (setDateAndTime() != 0) {
			cgiHeaderLocation("../datetimechangeerror.html");
		}
	} else if (cgiFormSubmitClicked("AbortRescue") == cgiFormSuccess) {
		cgiHeaderLocation("../index.html");
	} else if (cgiFormSubmitClicked("ContinueRescue") == cgiFormSuccess) {
		unlink(UPLOAD_STATUS_FILE);
		if (inSetupMode()) {     // 12-30-2014 JOA - Modified
			RescueModule();
		} else {
			writeStatusFile("RESCUE_SETUP_NEEDED");
		}
	} else if (cgiFormSubmitClicked("RescueReboot") == cgiFormSuccess) {
			sleep(2);
			shutdownSystem();
	}
	else {
	// if no submitted option, redirect to the main page
	cgiHeaderLocation("../index.html");
	}
	return( 0 );
}


